******************************************
* This Stata do-file is written to accompany the paper:
* Amy King & Andrew Leigh, "Bias at the Ballot Box? Testing Whether
* Candidates Gender Affects Their Vote", Social Science Quarterly, 2010
* Feel free to adapt this code, but please cite our paper if you do so.
* 
* The first part of the dofile combines data from several disparate datasets.
* Some of these are confidential, so I have not included them in the zipfile.
* To run the dofile, please start at line 310. The code up to that point is just background.
******************************************

version 10
clear
set more off
set mem 100m
set matsize 800

******************************************
* Constructing a measure of gender attitudes from 1987-2004
******************************************
use "C:\Users\aleigh\Datasets\Australian Election Studies\AES1987\aes1987.dta", clear
ren q51eqopw eqopp1987
keep eqopp*
append using "C:\Users\aleigh\Datasets\Australian Election Studies\AES1990\aes1990.dta"
ren f5equop eqopp1990
keep eqopp*
append using  "C:\Users\aleigh\Datasets\Australian Election Studies\AES1993\aes1993.dta" 
ren e2equop eqopp1993
keep eqopp*
append using "C:\Users\aleigh\Datasets\Australian Election Studies\AES1996\aes1996.dta" 
ren e2equop eqopp1996
keep eqopp*
append using "C:\Users\aleigh\Datasets\Australian Election Studies\AES1998\aes1998.dta" 
ren e2equop eqopp1998
keep eqopp*
append using "C:\Users\aleigh\Datasets\Australian Election Studies\AES2001\aes2001.dta" 
ren e2equop eqopp2001
keep eqopp*
append using "C:\Users\aleigh\Datasets\Australian Election Studies\AES2004\d1079.dta" 
ren e2equop eqopp2004
keep eqopp*
codebook
recode eqopp1987 1=1 2/3=0
for num 1990 1993 1996 1998 2001 2004: recode eqoppX 1/2=1 3/5=0
label drop _all
gen n=_n
reshape long eqopp, i(n) j(election)
drop n
collapse eqopp,by(election)
la var eqopp "Share who say that equal opportunity for women has gone too far"
sort election
cd "C:\Users\aleigh\My publications\Aust - beautiful politicians etc\Gender bias\"
save temp_eqopp, replace
merge election using gender_pay_gap, nokeep
*set scheme s1mono
*tw line eq election,lpattern(dash) lcolor(black) || line g election,lpattern(solid) lcolor(black) yaxis(2) yti("Share who say that equal opportunity for" "women has gone too far (dashed)",axis(1)) yti("Gender pay gap (solid)",axis(2)) xti("") legend(off)
corr eqopp gen

***************************************************
* Merging datasets together
***************************************************
cd "C:\Users\aleigh\Datasets\AEC data\"
use 1901_2004_gender_race.dta, clear  
keep if election>=1984
save temp1984_2004, replace

use 1901_2004_gender_race.dta, clear  
keep if election<=1983
keep  firstname lastname electorate election gender first 
sort election electorate firstname lastname 
save temp1901_1983, replace

cd "C:\Users\aleigh\Datasets\AEC data\Stata\"
use clean_1901_1983,clear
cd "C:\Users\aleigh\Datasets\AEC data\"
sort election electorate firstname lastname 
merge election electorate firstname lastname using temp1901_1983, nokeep
tab _merge
drop _merge
append using temp1984_2004

**** Check unmerged
sort firstname lastname election electorate 
merge firstname lastname election electorate using gender_ethnicity_missing_1901_2004, update replace nokeep
drop _merge

list firstname lastname if gender==.

* Merging on incumbent data
replace incumbent=. if election<=2001
replace lastname=proper(lastname)
replace lastname=subinstr(lastname,"*","",.)
sort election electorate lastname
cd "C:\Users\aleigh\Datasets\AEC data\Incumbents\" 
merge election electorate lastname using incumbent_for_merge, update nokeep
tab _merge if election==1980
drop _merge
recode incumbent .=0

* Recoding incorrect incumbents
for ! in any Harry Arthur "Edward Charles" "Charles Edgar" "Allan Geoffrey" "David Graeme" "John Kenneth James" Kenneth Paul Silvia "Michael John" Shireen  \ @ in any Green Fenton Hill Abbott Brennan Morris Stewart Johnson Keating Smith Anderson Murphy  \ # in any Richmond Maribyrnong Echuca "New England" Batman Shortland Lang Hughes Blaxland Bass Gwydir Lowe  \ $ in num 1925 1928 1929 1943 1946 1974 1975 1977 1987 1990 2004 2004: replace incumbent=0 if firstname=="!" & lastname=="@" & electorate=="#" & election==$ 

* Merging in the gender earnings penalty data & equal opportunity attitudes
cd "C:\Users\aleigh\My publications\Aust - beautiful politicians etc\Gender bias\"
sort election
merge election using gender_pay_gap, nokeep
tab _merge
drop _merge
replace genderearningspenalty =-genderearningspenalty 
sort election
merge election using temp_eqopp
tab _merge
drop _merge

gen uniqueid=_n

* Calculating voteshare
bysort election electorate: egen totalvote=sum(primaryvote)
gen voteshare=primaryvote/totalvote
gen logvoteshare=ln(voteshare)

* Generating election dummies
egen electionno=group(election) 

* Recoding party indicators
replace party="LP" if (party=="LIB" | party=="LP/CP") & election>=1963
replace party="NCP" if party=="NP"

* Generating party dummies, with liberal party as base
replace party="IND" if party=="" 
gen ALP=1 if party=="ALP" 
mvencode ALP, mv(0)
gen nationalparty=1 if party=="NP"
mvencode nationalparty, mv(0)
gen democrats=1 if party=="DEM"
mvencode democrats, mv(0)
gen greens=1 if party=="GRN"
mvencode greens, mv(0)

* Minor party is a party that won <50% of primary vote in all seats
* Independent candidates are always minor
bysort election party: egen maxvote=max(voteshare)
gen minorparty=1 if maxvote<.5
recode minorparty .=0
replace minorparty=1 if party=="IND" 
bysort election: tab party if minorparty==0

* Generating sex
gen female=1 if gender==1
replace female=0 if gender==0
* For some reason, a couple of candidates did not have gender coded
list firstname election if female==.
replace female=0 if (firstname=="C. Murray" | firstname=="Lance Clyde" | firstname=="Daas" | firstname=="Wiyendji N. Jacob" | firstname=="Lance" | firstname=="E Charles" | firstname=="Larry" | firstname=="Daas" | firstname=="J S Tiger" | firstname=="C Duncan" | firstname=="Lance Neville" | firstname=="Cam" | firstname=="Beville" | firstname=="T John" | firstname=="Cy") & female==.
replace female=0 if (firstname=="Paris" | firstname=="Galloway" | firstname=="Elson Verco" | firstname=="Glenister" | firstname=="Guenter" | firstname=="Haset" | firstname=="Sol" | firstname=="Kusala Sujatha" | firstname=="Wajde" | firstname=="Makere") & female==.
replace female=1 if (firstname=="Zona" | firstname=="Loma" | firstname=="Lali") & female==.

* Fixing 2 miscodes
replace female=0 if firstname=="Lucien Lawrence" & lastname=="Cunningham" 
replace female=0 if firstname=="Millice" & lastname=="Culpin"

* Generating interaction variables
gen post1994=1 if election>=1994
replace post1994=0 if election<1994
gen post2002=1 if election>=2002
replace post2002=0 if election<2002
gen female_ALP94=female*ALP*post1994
gen female_ALP02=female*ALP*post2002
gen female_ALP=female*ALP
gen female_nationalparty=female*nationalparty
gen female_democrats=female*democrats
gen female_greens=female*greens
gen female_minorparty=female*minorparty

**************************************************************
* We set the election interaction so that it equals 1 in 1901
**************************************************************

* Generating interaction variables for linear time trends
gen female_election=female*(election-1900)

* Generating an election*party effect. This allows for party-specific swings in all elections.
egen election_party=group(electionno party)
egen election_electorate=group(election electorate)

**************************************************************
* Generating a lagged party*electorate voteshare measure
**************************************************************
egen party_electorate=group(electorate party)
egen minorparty_electorate=group(electorate minorparty)
egen candidateno=group(firstname lastname)
gen expected_voteshare=.
gen expected_voteshare_method=.

levelsof electionno, local(electionno)
foreach e of local electionno {
	
	* Replace with same party, previous election
	bysort party_electorate: egen temp1=mean(voteshare) if electionno==`e'-1
	bysort party_electorate: egen temp2=max(temp1) 
	bysort party_electorate: egen temp3=max(candidateno) if electionno==`e'-1
	bysort party_electorate: egen temp4=max(temp3) 
	replace expected_voteshare=temp2 if electionno==`e' & candidateno~=temp4 & expected_voteshare==.
	replace expected_voteshare_method=1 if expected_voteshare~=.
	drop temp*
	
	* Replace with same party, election-2, election-3, .... election-5
	bysort party_electorate: egen temp1=mean(voteshare) if electionno==`e'-2
	bysort party_electorate: egen temp2=max(temp1) 
	bysort party_electorate: egen temp3=max(candidateno) if electionno==`e'-2
	bysort party_electorate: egen temp4=max(temp3) 
	replace expected_voteshare=temp2 if electionno==`e' & candidateno~=temp4 & expected_voteshare==.
	drop temp*
	bysort party_electorate: egen temp1=mean(voteshare) if electionno==`e'-3
	bysort party_electorate: egen temp2=max(temp1) 
	bysort party_electorate: egen temp3=max(candidateno) if electionno==`e'-3
	bysort party_electorate: egen temp4=max(temp3) 
	replace expected_voteshare=temp2 if electionno==`e' & candidateno~=temp4 & expected_voteshare==.
	drop temp*
	bysort party_electorate: egen temp1=mean(voteshare) if electionno==`e'-4
	bysort party_electorate: egen temp2=max(temp1) 
	bysort party_electorate: egen temp3=max(candidateno) if electionno==`e'-4
	bysort party_electorate: egen temp4=max(temp3) 
	replace expected_voteshare=temp2 if electionno==`e' & candidateno~=temp4 & expected_voteshare==.
	drop temp*
	bysort party_electorate: egen temp1=mean(voteshare) if electionno==`e'-5
	bysort party_electorate: egen temp2=max(temp1) 
	bysort party_electorate: egen temp3=max(candidateno) if electionno==`e'-5
	bysort party_electorate: egen temp4=max(temp3) 
	replace expected_voteshare=temp2 if electionno==`e' & candidateno~=temp4 & expected_voteshare==.
	drop temp*
	replace expected_voteshare_method=2 if expected_voteshare~=. & expected_voteshare_method==.

	* Replace with average major/minor party vote for that electorate in previous election
	bysort minorparty_electorate: egen temp1=mean(voteshare) if electionno==`e'-1
	bysort minorparty_electorate: egen temp2=max(temp1) 
	replace expected_voteshare=temp2 if electionno==`e' & expected_voteshare==.
	drop temp*
	replace expected_voteshare_method=3 if expected_voteshare~=. & expected_voteshare_method==.

	* Replace with average national vote for same party in previous election
	bysort party: egen temp1=mean(voteshare) if electionno==`e'-1
	bysort party: egen temp2=max(temp1) 
	replace expected_voteshare=temp2 if electionno==`e' & expected_voteshare==.
	drop temp*
	replace expected_voteshare_method=4 if expected_voteshare~=. & expected_voteshare_method==.

	* Replace with average national vote for major/minor party in previous election
	bysort minorparty: egen temp1=mean(voteshare) if electionno==`e'-1
	bysort minorparty: egen temp2=max(temp1) 
	replace expected_voteshare=temp2 if electionno==`e' & expected_voteshare==.
	drop temp*
	replace expected_voteshare_method=5 if expected_voteshare~=. & expected_voteshare_method==.

}

tab election if expected_voteshare==.
tab expected_voteshare_method
bysort minorparty female: tab expected_voteshare_method
drop expected_voteshare_method

********************************************
* Female share variables
* Constructed so that they don't include the individual
********************************************
bysort election_electorate: egen temp1=sum(female) 
replace temp1=temp1-female if female~=.
bysort election_electorate: egen temp2=count(female) 
replace temp2=temp2-1 if female~=.
gen fs_electorate=temp1/temp2
drop temp*
bysort election: egen temp1=sum(female) 
replace temp1=temp1-female if female~=.
bysort election: egen temp2=count(female) 
replace temp2=temp2-1 if female~=.
gen fs_election=temp1/temp2
drop temp*
gen female_incumbent=female*incumbent
bysort election: egen temp1=sum(female_incumbent) 
replace temp1=temp1-female_incumbent if female_incumbent~=.
bysort election: egen temp2=sum(incumbent)
replace temp2=temp2-incumbent if incumbent~=.
gen fs_incumbents=temp1/temp2
replace fs_incumbents=0 if election==1901
drop temp* 
sum fs*
sum fs* if election==2004

* Constructing %female candidates/incumbents measures (for graphs)
bysort election: egen temp1=sum(female) 
bysort election: egen temp2=count(female) 
gen fpct_candidates=temp1/temp2
drop temp*
bysort election: egen temp1=sum(female_incumbent) 
bysort election: egen temp2=sum(incumbent)
gen fpct_incumbents=temp1/temp2
replace fpct_incumbents=0 if election==1901
drop temp* female_incumbent

* Total candidates measure
bysort election_electorate: egen totalcand=count(female)
replace totalcand=1/totalcand

drop ethnicity

cd "C:\Users\aleigh\My publications\Aust - beautiful politicians etc\Gender bias\"
save 1901_2004_gender_web, replace

****************************************
* End of setup. Start running the file from here.
****************************************
cd "C:\Users\aleigh\My publications\Aust - beautiful politicians etc\Gender bias\"
use 1901_2004_gender_web, clear

****************************************
* Pretty graphs
****************************************
set scheme s1mono
sort election
tw line fpct_candidates election, xti("") yti("") xlabel(,grid) legend(off) ti("Figure 1: Share of candidates who were women") 
for num 1: graph export figX.wmf,replace \ graph export figX.eps,replace \ graph save figX.gph,replace
tabstat fpct_candidates fpct_incumbents ,by(election)
sort election
#delimit ;
tw line fpct_incumbents election, xti("") yti("") xlabel(,grid) legend(off) ti("Figure 2: Share of incumbents who were women") 
text(0 1901 "1902: Women eligible to vote in and contest federal elections",orientation(vertical) place(n) size(small))
text(0 1904 "1903: Four women contest the Federal Election",orientation(vertical) place(n) size(small))
text(0 1921 "1921: First woman elected to a state parliament (WA)",orientation(vertical) place(n) size(small))
text(.01 1943 "1943: Two women elected to Federal Parliament",orientation(vertical) place(n) size(small))
text(.03 1949 "1949: First woman appointed to Cabinet rank",orientation(vertical) place(n) size(small))
text(0 1966 "1966: First woman appointed to run a government department",orientation(vertical) place(n) size(small))
text(0 1973 "1973: First adviser to the PM on issues affecting women",orientation(vertical) place(n) size(small))
text(.09 1983 "1984: Sex Discrimination Act",orientation(vertical) place(n) size(small))
text(.09 1986.5 "1986: First female speaker of Reps",orientation(vertical) place(n) size(small))
text(.09 1989 "& first female party leader (Democrats)",orientation(vertical) place(n) size(small));
#delimit cr
for num 2: graph export figX.wmf,replace \ graph export figX.eps,replace \ graph save figX.gph,replace

****************************************
* Summary statistics
****************************************
reg voteshare female incumbent expected_voteshare totalcand, cl(election_electorate)
sum voteshare female expected_voteshare incumbent totalcand femvotshare fs* fpct* genderearningspenalty if e(sample), sep(0)
codebook candidateno if e(sample)
codebook election_electorate if e(sample)

****************************************
* Gender Regressions
****************************************
cd "C:\Users\aleigh\My publications\Aust - beautiful politicians etc\Gender bias\"
reg voteshare female incumbent expected_voteshare totalcand, cl(election_electorate)
outreg using results.doc, coefastr nocons bracket 3aster replace bdec(3) se ct("No FE")
xi: areg voteshare female incumbent expected_voteshare totalcand i.party, a(electionno) cl(election_electorate)
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Election & Party FE")
areg voteshare female incumbent expected_voteshare totalcand, a(election_party) cl(election_electorate)
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Election*Party FE")
areg voteshare female incumbent expected_voteshare totalcand , a(election_party) cl(election_electorate)

*******************************************
* The preselection problem
*******************************************
for num 0 1: areg voteshare female expected_voteshare totalcand if incumbent==X & minorparty==0, a(election_party) cl(election_electorate) \ outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Incumbent=X; Election*Party FE") \ sum voteshare if e(sample) 
gen independent=1 if party=="IND" 
recode independent .=0
for num 0/1: areg voteshare female incumbent expected_voteshare totalcand if independent==X, a(election_party) cl(election_electorate) \ outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Independent=X; Election*Party FE") \ sum voteshare if e(sample) 

*****************************************
* How does it look across parties?
*****************************************
bysort party: egen party_candidates=sum(female)
gen str party_big=party if party_candidates>100 & party_candidates<.
levelsof party_big, local(party_big)
foreach p of local party_big {
	gen femalep_`p'=female if party_big=="`p'"
	recode femalep_`p' .=0
	} 
tabstat party_candidates,by(party_big)
xi: areg voteshare incumbent expected_voteshare totalcand femalep_*, a(election_party) cl(election_electorate)
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("By Party - Election*Party FE") 
gen janine_haines=1 if firstname=="Janine" & lastname=="Haines"
xi: areg voteshare incumbent expected_voteshare totalcand femalep_* if janine_haines~=1, a(election_party) cl(election_electorate)


*******************************************
* How does the gender effect change over time?
*******************************************
levelsof election, local(election)
foreach e of local election {
	gen female_`e'=female if election==`e'
	recode female_`e' .=0
	}
xi: areg voteshare incumbent expected_voteshare totalcand female_1901-female_2004, a(election_party) cl(election_electorate)
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("By Year - Election*Party FE") 
gen female_interaction=.
gen female_ub=.
gen female_lb=.
levelsof election, local(election)
foreach e of local election {
	replace female_interaction=_b[female_`e'] if election==`e'
	replace female_lb=_b[female_`e']+1.6*_se[female_`e'] if election==`e'
	replace female_ub=_b[female_`e']-1.6*_se[female_`e'] if election==`e'
}
sort election
#delimit ; 
tw line female_interaction election if election>=1903, lpattern(solid) lcolor(black) yaxis(1) yti("Penalty against female candidates (line)",axis(1)) || 
	line female_ub election if election>=1903, lpattern(dash) yaxis(1) lcolor(black) || 
	line female_lb election if election>=1903, lpattern(dash) yaxis(1) lcolor(black) || 
	bar fs_election election if election>=1903, yaxis(2)  xlabel(,grid) yti("Share of candidates who were female (bars)",
	axis(2)) ysc(r(0 1) axis(2)) ylabel(#5,axis(2)) yline(0) xti("") legend(off) ti("Figure 3: Ballot box penalty against women (1903-2004)")
	subti("Solid line denotes penalty, dashed lines denote 90% confidence interval." "Bars denote share of candidates who were women.");
#delimit cr
for num 3: graph export figX.wmf,replace \ graph export figX.eps,replace \ graph save figX.gph,replace
tabstat female_interaction, by(election)

sort election
#delimit ; 
tw line female_interaction election if election>=1980, lpattern(solid) lcolor(black) yaxis(1) yti("Penalty against female candidates (line)",axis(1)) || 
	line female_ub election if election>=1980, lpattern(dash) lcolor(black) yaxis(1) || 
	line female_lb election if election>=1980, lpattern(dash) lcolor(black) yaxis(1) || 
	bar fs_election election if election>=1980, yaxis(2)  xlabel(,grid) yti("Share of candidates who were female (bars)",
	axis(2)) ysc(r(0 1) axis(2)) ylabel(#5,axis(2)) yline(0) xti("") legend(off) ti("Figure 4-DP: Ballot box penalty against women (1980-2004)")
	subti("Solid line denotes penalty, dashed lines denote 90% confidence interval." "Bars denote share of candidates who were women.");
#delimit cr
egen tag=tag(election)
tabstat female_interaction if election>=1980 & tag,by(election)
drop tag

sort election
#delimit ; 
tw line female_interaction election if election>=1903, yaxis(1) lcolor(black) yti("Penalty against female candidates (thin line)",axis(1)) ysc(r(-.35 .1) axis(1)) || 
	line female_ub election if election>=1903, lpattern(dash) lcolor(black) yaxis(1) || 
	line female_lb election if election>=1903, lpattern(dash) lcolor(black) yaxis(1) || 
	line genderearningspenalty election if election>=1903, lwidth(thick) yaxis(2)  xlabel(,grid) yti("Gender pay gap (thick line)",axis(2)) ysc(r(-.7 .2) axis(2) ) 
	ylabel(-.6 -.4 -.2 0 .2,axis(2)) yline(0) xti("") legend(off) ti("Figure 4: The ballot box and labor market" "penalty against women")
	subti("Thin line denotes penalty, dashed lines denote 90% confidence" "interval, thick line denotes gender pay gap.");
#delimit cr
for num 4: graph export figX.wmf,replace \ graph export figX.eps,replace \ graph save figX.gph,replace

#delimit ; 
tw scatter voteshare femvot if female==1 || lfit voteshare femvot if female==1  , 
xti("Share of electorate that is female") yti("Voteshare of female candidate") legend(off) ti("Figure 6-DP: Female candidates and female voters (1903-66)") 
subti("Each dot denotes a different candidate");
#delimit cr
xi: areg voteshare femvot incumbent expected_voteshare totalcand if female==1, a(election) cl(election_electorate)
xi: areg voteshare femvot incumbent expected_voteshare totalcand if female==1, a(election_party) cl(election_electorate)
sum femvot if female==1

for var fs* genderearningspenalty femvotshare fpct*: gen fX=female*X

* Interactions with female candidate share - explaining time series
for var fpct_incumbents fpct_candidates genderearningspenalty: areg voteshare female incumbent expected_voteshare totalcand fX , a(election_party) cl(election_electorate) \ outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Interactions: Election*Party FE") 
areg voteshare female incumbent expected_voteshare totalcand ffpct_incumbents ffpct_candidates fgenderearnings , a(election_party) cl(election_electorate) 
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Explaining time series: Election*Party FE") 
* Robustness checks on column 4 result
areg voteshare female incumbent expected_voteshare totalcand ffpct_incumbents fgenderearnings , a(election_party) cl(election_electorate) 
areg voteshare female incumbent expected_voteshare totalcand ffpct_incumbents ffpct_candidates , a(election_party) cl(election_electorate) 
corr fpct_incumbents fpct_candidates genderearnings if e(sample)
corr fpct_incumbents fpct_candidates genderearnings eqopp if e(sample)
gen feqopp=female*-eqopp
areg voteshare female incumbent expected_voteshare totalcand ffpct_incumbents ffpct_candidates feqopp, a(election_party) cl(election_electorate) 
areg voteshare female incumbent expected_voteshare totalcand ffpct_incumbents ffpct_candidates fgenderearnings if eqopp~=., a(election_party) cl(election_electorate) 
areg voteshare female incumbent expected_voteshare totalcand ffpct_incumbents ffpct_candidates fgenderearnings if election<1980, a(election_party) cl(election_electorate) 
areg voteshare female incumbent expected_voteshare totalcand ffpct_incumbents ffpct_candidates fgenderearnings if election>=1980, a(election_party) cl(election_electorate) 

* Interactions with female candidate share - intra-electorate effects
xi: areg voteshare female incumbent expected_voteshare totalcand ffs_electorate fs_electorate i.female*i.election, a(election_party) cl(election_electorate) 
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Intra-electorate gender effects: Election*Fem & Election*Party FE") 
xi: areg voteshare female incumbent expected_voteshare totalcand ffs_electorate fs_electorate i.female*i.election if femvotshare~=., a(election_party) cl(election_electorate) 
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Intra-electorate gender effects: Election*Fem & Election*Party FE (1903-66 sample)") 
xi: areg voteshare female incumbent expected_voteshare totalcand ffemvotshare femvotshare i.female*i.election, a(election_party) cl(election_electorate) 
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Intra-electorate gender effects: Election*Fem & Election*Party FE") 
xi: areg voteshare female incumbent expected_voteshare totalcand ffemvotshare ffs_electorate femvotshare fs_electorate i.female*i.election, a(election_party) cl(election_electorate) 
outreg using results.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Intra-electorate gender effects: Election*Fem & Election*Party FE") 

* ALP post-1994/2002 Regression
xi: areg voteshare female_ALP94 incumbent expected_voteshare totalcand i.female*i.party i.female*i.election, a(election_party) cl(election_electorate)
outreg using results_alp.doc, coefastr nocons bracket 3aster replace bdec(3) se ct("female*party, female*election, election*party FE")
xi: areg voteshare female_ALP02 incumbent expected_voteshare totalcand i.female*i.party i.female*i.election, a(election_party) cl(election_electorate)
outreg using results_alp.doc, coefastr nocons bracket 3aster append bdec(3) se ct("female*party, female*election, election*party FE")

* Program to calculate p-values from a 2-sample t-test
program drop _all
program define pvalues
set output error
clear
set obs 1
/*** create t-statistic for difference between 2 means ***/
gen diff=$mean1-$mean2
gen t = ($mean1-$mean2)/sqrt((($semean1*sqrt($n1))^2/$n1)+(($semean2*sqrt($n2))^2/$n2))
/*** create 2-tailed probability for t-statistic ***/
gen p = 2*(1-normal(abs(t)))
set output proc
sum diff t p 
end

* Test of difference - Incumbents & NonIncumbents
for num 5905 -.018 .004 3099 -.010 .005 \ any n1 mean1 semean1 n2 mean2 semean2: set output error \ global Y X
pvalues

* Test of difference - Independents & NonIndependents
for num 14853 -.006 .001 1914 -.006 .002 \ any n1 mean1 semean1 n2 mean2 semean2: set output error \ global Y X
pvalues

